function [vertices_mat, edges_mat, faces_mat, bodies_mat, errcheck] = f3_read_se_shape_file(shape_paths)
% Description
% - Finds lists of coordinates exported from Surface Evolver in a folder ~\shapes\source\
%   prefix_vertices.txt
%   prefix_faces.txt
%   prefix_edges.txt
%   prefix_bodies.txt
% - Extracts all coordinates and stores them as 2D matrices
% - Saves all coordinates in '~\shapes\source\shape_data.mat'
%
%   Dmitry Ershov
%   2017 October

% OUTPUT:
vertices_mat    = [];   % matrix containing vertex info: [v_id, x, y, z]
edges_mat       = [];   % matrix containing edges  info: [e_id, v1_id, v2_id]
faces_mat       = [];   % matrix containing faces  info: [f_id, e1_id, e2_id, e3_id]
bodies_mat      = [];   % matrix containing bodies info: [b_id, f1_id, f2_id, ..., fn_id]
errcheck        = 0;    % boolean if there is any Nan in the Matrices. 



[filename, path_name] = uigetfile([shape_paths.shapes_source, '*.*'], 'Select a source SE file (original; not the exported lists!)');
file_abs_path = [path_name, filename];

if filename == 0
    disp(' ')
    disp('No file was selected')
    return;
end



%% 1. Find Vertices
disp('  ')
disp(['Extracting vertices...'])

% format of the vertex table
readFormat          = '%f64 %f64 %f64 %f64 %*[^\n]';            

% open file, skip the first line
source_file_path    =  [file_abs_path, '_vertices', '.txt'];
fid                 = fopen(source_file_path);
fgetl(fid); 

% scan the txt file.
M                   = textscan(fid, readFormat);
% compile matrix
vertices_mat        = [M{1}, M{2}, M{3}, M{4}];                
% close the file
fclose(fid);


disp('Found vertices:  ')
disp(sprintf('%10s\t%10s\t%10s\t%10s', 'Id','x','y','z'))
disp(sprintf('%10.30g \t', vertices_mat(1,:)))
disp(sprintf('%10s\t%10s\t%10s\t%10s', ':',':',':',':'))
% disp('    ............................................................')
disp(sprintf('%10.30g \t', vertices_mat(end,:)))

if any(any(isnan(vertices_mat)))
    vertices_mat(isnan(vertices_mat)) = [];
    disp(' ')
    disp('There were NaNs in the vertices matrix. Removed.');
    errcheck = 1;
end



%% 2. Find edges
disp(' ')
disp('Extracting edges...')

% format of the vertex table
readFormat          = '%f64 %f64 %f64 %*[^\n]';            

% open file, skip the first line
source_file_path    =  [file_abs_path, '_edges', '.txt'];
fid                 = fopen(source_file_path);
fgetl(fid); 

% scan the txt file.
M                   = textscan(fid, readFormat);
% compile matrix
edges_mat        = [M{1}, M{2},  M{3}];                
% close the file
fclose(fid);

disp('Found edges:  ')
disp(sprintf('%10s\t%10s\t%10s\t', 'Id','endpoint_1','endpoint_2'))
disp(sprintf('%10.30g \t', edges_mat(1,:)))
disp(sprintf('%10s\t%10s\t%10s\t', ':',':',':'))
% disp('    ............................................................')
disp(sprintf('%10.30g \t', edges_mat(end,:)))

if any(any(isnan(edges_mat)))
    edges_mat(isnan(edges_mat)) = [];
    disp(' ')
    disp('There were NaNs in the edges matrix. Removed.');
    errcheck = 1;
end




%% 3. Find faces
disp(' ')
disp('Extracting faces...')

% format of the vertex table
readFormat          = '%f64 %f64 %f64 %f64 %*[^\n]';            

% open file, skip the first line
source_file_path    = [file_abs_path, '_faces', '.txt'];
fid                 = fopen(source_file_path);
fgetl(fid); 

% scan the txt file.
M                   = textscan(fid, readFormat);
% compile matrix
faces_mat        = [M{1}, M{2},  M{3}, M{4}];                
% close the file
fclose(fid);

disp('Found faces:  ')
disp(sprintf('%10s\t%10s\t%10s\t%10s', 'Id','edge_1', 'edge_2', 'edge_3'))
disp(sprintf('%10.30g \t', faces_mat(1,:)))
disp(sprintf('%10s\t%10s\t%10s\t%10s', ':',':',':',':'))
% disp('    ............................................................')
disp(sprintf('%10.30g \t', faces_mat(end,:)))

if any(any(isnan(faces_mat)))
    faces_mat(isnan(faces_mat)) = [];
    disp(' ')
    disp('There were NaNs in the faces matrix. Removed.');   
    errcheck = 1;
end




%% 4. Find bodies
disp(' ')
disp('Extracting bodies...')

% format of the vertex table
readFormat_1          = '%f64 %f64 %f64 %f64 %f64 %f64 %f64 %f64  %f64 %f64 %f64 %*[^\n]';            
readFormat_2          = '%f64 %f64 %f64 %f64 %f64 %f64 %f64 %f64  %f64 %f64 %*[^\n]';            

bodies_mat = [];

% open file, skip the first line
done = 0;
body_ind = 0;

while ~done
    
    body_ind            = body_ind + 1;
    
    source_file_path    = [file_abs_path, '_body_', num2str(body_ind), '.txt'];
    fid                 = fopen(source_file_path);
    
    if fid == -1
       disp(['Body ' num2str(body_ind) ' was not found.'])
       done = 1;
       continue;
    end
    
    fgetl(fid); 

    % read first line with format 1
    M_1                 = textscan(fid, readFormat_1, 1);
    M_1(1)              = []; % this is a body index; remove it.
    body_faces          = cell2mat(M_1); % collect faces from the first line
    body_faces          = body_faces(:);

    % read all the rest lines with format 2
    M_2                   = textscan(fid, readFormat_2);
    fclose(fid);


    % collect all body faces that were found during scanning. 
    M_3 = M_2{1};
    for i = 2 : size(M_2, 2)    
        if size(M_2{i},1) == size(M_3, 1) % if equal, collect without changes
            M_3 = [M_3, M_2{i}];
        elseif size(M_2{i},1) < size(M_3, 1) % if shorter, append 0 and collect
            M_3 = [M_3, [M_2{i}; 0]];        
        end
        %body_faces = [body_faces; M_2{i}];
    end

    M_3                 = M_3';
    M_3                 = M_3(:);   % make a column. 
    % Transpose - because of display purposes. It is easier to compare to the raw data, if
    % Column vector is done not as concatenation of column-to-column, but
    % as concatenation of row-to-row.

    M_3(M_3 == 0)       = [];       % remove zeroes.
    body_faces          = [body_faces; M_3(:)]; 
    body_ind_mat        = [repmat(body_ind, size(body_faces, 1), 1), body_faces];   % concatenate
    bodies_mat          = [bodies_mat; body_ind_mat];    
end


disp('Found bodies:')
disp(sprintf('%10s\t%10s', 'Id','face'))
for i = 1:4
    disp(sprintf('%10.15g \t', bodies_mat(i, :)))
end
disp(sprintf('%10s\t%10s\t', ':',':'))
disp(sprintf('%10s\t%10s\t', ':',':'))
for i = -3 : 1 : 0
    disp(sprintf('%10.15g \t', bodies_mat(end + i, :)))
end

if any(any(isnan(bodies_mat)))
    bodies_mat(isnan(bodies_mat)) = [];
    disp(' ')
    disp('There were NaNs in the bodies matrix. Removed.');    
    errcheck = 1;  
end

